<?php
// /public_html/api/_bootstrap.php

declare(strict_types=1);

error_reporting(E_ALL);
ini_set('display_errors', '0');

require_once __DIR__ . '/../connect.php'; // defines $con (mysqli)

if (!isset($con) || !($con instanceof mysqli)) {
  http_response_code(500);
  echo 'Missing $con mysqli connection';
  exit;
}

function json_out(array $data, int $code = 200): void {
  if (function_exists('ob_get_level')) { while (ob_get_level()) { @ob_end_clean(); } }
  http_response_code($code);
  header('Content-Type: application/json; charset=UTF-8');
  echo json_encode($data, JSON_UNESCAPED_SLASHES);
  exit;
}

function require_method(string $method): void {
  $m = $_SERVER['REQUEST_METHOD'] ?? 'GET';
  if (strtoupper($m) !== strtoupper($method)) {
    json_out(['success' => false, 'message' => "Method not allowed. Use $method."], 405);
  }
}

function read_json_body(): array {
  $raw = file_get_contents('php://input');
  if ($raw === false || trim($raw) === '') return [];
  $data = json_decode($raw, true);
  if (!is_array($data)) json_out(['success' => false, 'message' => 'Invalid JSON body.'], 400);
  return $data;
}

function str_or(string $v, string $default=''): string { return trim($v) !== '' ? trim($v) : $default; }

function uuid_v4(): string {
  $data = random_bytes(16);
  $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
  $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);
  $hex = bin2hex($data);
  return sprintf('%s-%s-%s-%s-%s',
    substr($hex, 0, 8),
    substr($hex, 8, 4),
    substr($hex, 12, 4),
    substr($hex, 16, 4),
    substr($hex, 20, 12)
  );
}

// ---------- Upload configuration ----------
const UPLOAD_ROOT = __DIR__ . '/../uploads'; // /public_html/uploads
const URL_UPLOADS_PREFIX = '/uploads';       // served via web root

function ensure_dir(string $absDir): void {
  if (!is_dir($absDir)) {
    if (!@mkdir($absDir, 0755, true) && !is_dir($absDir)) {
      json_out(['success' => false, 'message' => 'Failed to create upload directory.'], 500);
    }
  }
}

function safe_filename(string $name): string {
  $name = preg_replace('/[^a-zA-Z0-9._-]+/', '_', $name);
  $name = trim($name, '._-');
  return $name !== '' ? $name : 'file';
}

function detect_mime(string $tmpPath): string {
  $fi = new finfo(FILEINFO_MIME_TYPE);
  $mime = $fi->file($tmpPath);
  return is_string($mime) ? $mime : 'application/octet-stream';
}

function move_uploaded_file_validated(
  array $file,
  array $allowedMimes,
  string $destAbsPath
): array {
  if (!isset($file['error']) || $file['error'] !== UPLOAD_ERR_OK) {
    return ['ok' => false, 'message' => 'Upload failed (error code).'];
  }
  if (!is_uploaded_file($file['tmp_name'])) {
    return ['ok' => false, 'message' => 'Invalid upload source.'];
  }

  $mime = detect_mime($file['tmp_name']);
  if (!in_array($mime, $allowedMimes, true)) {
    return ['ok' => false, 'message' => "Unsupported file type ($mime).", 'mime' => $mime];
  }

  ensure_dir(dirname($destAbsPath));

  if (!@move_uploaded_file($file['tmp_name'], $destAbsPath)) {
    return ['ok' => false, 'message' => 'Failed to move uploaded file.'];
  }

  return ['ok' => true, 'mime' => $mime];
}

function resolve_project_by_token(mysqli $con, string $token): array {
  $token = trim($token);
  if ($token === '') json_out(['success' => false, 'message' => 'Missing token.'], 400);

  $stmt = $con->prepare("SELECT id, project_token, name, notes, created_at, updated_at FROM projects WHERE project_token = ?");
  if (!$stmt) json_out(['success' => false, 'message' => 'DB prepare failed.'], 500);
  $stmt->bind_param('s', $token);
  $stmt->execute();
  $res = $stmt->get_result();
  $row = $res ? $res->fetch_assoc() : null;
  $stmt->close();

  if (!$row) json_out(['success' => false, 'message' => 'Project not found.'], 404);
  $row['id'] = (int)$row['id'];
  return $row;
}

function clamp_dec($v, float $min, float $max, float $fallback): float {
  if (!is_numeric($v)) return $fallback;
  $f = (float)$v;
  if ($f < $min) $f = $min;
  if ($f > $max) $f = $max;
  return $f;
}