<?php
// /public_html/api/projects/save_items.php
require_once __DIR__ . '/../_bootstrap.php';

require_method('POST');
$body = read_json_body();

$token = (string)($body['token'] ?? '');
$project = resolve_project_by_token($con, $token);
$project_id = (int)$project['id'];

$items = $body['items'] ?? [];
$deleted_ids = $body['deleted_ids'] ?? [];

if (!is_array($items)) $items = [];
if (!is_array($deleted_ids)) $deleted_ids = [];

$con->begin_transaction();

try {
  // Deletes
  if (!empty($deleted_ids)) {
    $ids = array_values(array_filter(array_map('intval', $deleted_ids), fn($v) => $v > 0));
    if (!empty($ids)) {
      // Build IN (...)
      $placeholders = implode(',', array_fill(0, count($ids), '?'));
      $types = str_repeat('i', count($ids) + 1);
      $sql = "DELETE FROM project_items WHERE project_id = ? AND id IN ($placeholders)";
      $stmt = $con->prepare($sql);
      if (!$stmt) throw new Exception('DB prepare failed (delete).');
      $params = array_merge([$project_id], $ids);

      // bind dynamically
      $bind = [];
      $bind[] = $types;
      foreach ($params as $k => $v) $bind[] = &$params[$k];
      call_user_func_array([$stmt, 'bind_param'], $bind);

      $stmt->execute();
      $stmt->close();
    }
  }

  // Inserts/Updates
  $inserted = 0;
  $updated  = 0;
  $id_map   = []; // temp client index => new id (optional)

  $stmtIns = $con->prepare("
    INSERT INTO project_items (project_id, asset_id, x_px, y_px, rotation_deg, z_index, locked, meta_json)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
  ");
  if (!$stmtIns) throw new Exception('DB prepare failed (insert).');

  $stmtUpd = $con->prepare("
    UPDATE project_items
    SET asset_id=?, x_px=?, y_px=?, rotation_deg=?, z_index=?, locked=?, meta_json=?
    WHERE id=? AND project_id=?
  ");
  if (!$stmtUpd) throw new Exception('DB prepare failed (update).');

  foreach ($items as $idx => $it) {
    if (!is_array($it)) continue;

    $id = (int)($it['id'] ?? 0);
    $asset_id = (int)($it['asset_id'] ?? 0);
    if ($asset_id <= 0) continue;

    $x = (float)($it['x_px'] ?? 0);
    $y = (float)($it['y_px'] ?? 0);
    $rot = (float)($it['rotation_deg'] ?? 0);
    $z = (int)($it['z_index'] ?? 0);
    $locked = (int)($it['locked'] ?? 0);
    $meta = $it['meta_json'] ?? null;

    $meta_json = null;
    if ($meta !== null) {
      $meta_json = json_encode($meta, JSON_UNESCAPED_SLASHES);
      if ($meta_json === false) $meta_json = null;
    }

    if ($id <= 0) {
      $stmtIns->bind_param('iiddidis',
        $project_id,
        $asset_id,
        $x,
        $y,
        $rot,
        $z,
        $locked,
        $meta_json
      );
      $stmtIns->execute();
      $newId = (int)$stmtIns->insert_id;
      $inserted++;
      $id_map[(string)$idx] = $newId; // optional
    } else {
      $stmtUpd->bind_param('iddidisiii',
        $asset_id,
        $x,
        $y,
        $rot,
        $z,
        $locked,
        $meta_json,
        $id,
        $project_id
      );
      $stmtUpd->execute();
      $updated++;
    }
  }

  $stmtIns->close();
  $stmtUpd->close();

  $con->commit();

  json_out([
    'success' => true,
    'inserted' => $inserted,
    'updated' => $updated,
    'id_map' => $id_map
  ]);

} catch (Throwable $e) {
  $con->rollback();
  json_out(['success' => false, 'message' => 'Save failed.', 'error' => $e->getMessage()], 500);
}